/**
 *  The package contains classes related for paging and page navigation support for JIDE Data Grids product.
 */
package com.jidesoft.paging;


/**
 *  <code>PageNavigationSupport</code> is an interface to support paging on a table model or a list model. It has methods
 *  to get/set the page size, the current page index, the total row count and methods to navigate the pages. It also
 *  supports <code>PageNavigationListener</code> which will tell you when the current page index changes, the page size
 *  changes or total page count changes.
 */
public interface PageNavigationSupport {

	/**
	 *  Gets the page size.
	 * 
	 *  @return the page size.
	 */
	public int getPageSize();

	/**
	 *  Sets the page size.
	 * 
	 *  @param pageSize the new page size.
	 */
	public void setPageSize(int pageSize);

	/**
	 *  Gets the current page.
	 * 
	 *  @return the current page.
	 */
	public int getCurrentPage();

	/**
	 *  Sets the current page.
	 * 
	 *  @param page the new current page.
	 */
	public void setCurrentPage(int page);

	/**
	 *  Gets the total page count.
	 * 
	 *  @return the total page count.
	 */
	public int getPageCount();

	/**
	 *  Gets the total record/row count.
	 * 
	 *  @return the total record/row count.
	 */
	public int getTotalRecordCount();

	/**
	 *  Sets the total record/row count.
	 * 
	 *  @param totalRecordCount the new total row count.
	 */
	public void setTotalRecordCount(int totalRecordCount);

	/**
	 *  Navigates to the next page.
	 */
	public void nextPage();

	/**
	 *  Navigates to the previous page.
	 */
	public void previousPage();

	/**
	 *  Navigates to the first page.
	 */
	public void firstPage();

	/**
	 *  Navigates to the last page.
	 */
	public void lastPage();

	/**
	 *  Adds a PageNavigationListener so that it will be notified when the page size changes, the current page index
	 *  changes or total page count changes.
	 * 
	 *  @param l a PageNavigationListener
	 */
	public void addPageNavigationListener(PageNavigationListener l);

	/**
	 *  Removes PageNavigationListener that was added before.
	 * 
	 *  @param l an existing PageNavigationListener.
	 */
	public void removePageNavigationListener(PageNavigationListener l);

	/**
	 *  Gets all the PageNavigationListeners that were added before.
	 * 
	 *  @return all the PageNavigationListeners.
	 */
	public PageNavigationListener[] getPageNavigationListeners();

	/**
	 *  Fires the PageNavigationEvent to all the PageNavigationListeners .
	 * 
	 *  @param source   the source of the event.
	 *  @param id       the event type. It could be {@link PageNavigationEvent#PAGE_COUNT_CHANGED},  {@link
	 *                  PageNavigationEvent#PAGE_SIZE_CHANGED}, or {@link PageNavigationEvent#CURRENT_PAGE_CHANGED}.
	 *  @param oldValue the old value. It could be the old page count, the old page size or the old current page index,
	 *                  depending on the value of the id parameter.
	 *  @param newValue the new value. It could be the new page count, the new page size or the new current page index,
	 *                  depending on the value of the id parameter.
	 */
	public void firePageNavigationEvent(Object source, int id, int oldValue, int newValue);
}
