/**
 *  The package contains classes for JIDE Pivot Grid product.
 */
package com.jidesoft.pivot;


/**
 *  This is an interface for PivotDataModel. We have a default implementation, which is {@link
 *  com.jidesoft.pivot.PivotDataModel}. However, with this interface, you can create your own PivotDataModel if you
 *  want.
 */
public interface IPivotDataModel extends PivotConstants {

	/**
	 *  Sets the data source.
	 * 
	 *  @param dataSource the new data source.
	 */
	public void setDataSource(PivotDataSource dataSource);

	/**
	 *  Gets the data source.
	 * 
	 *  @return the data source.
	 */
	public PivotDataSource getDataSource();

	/**
	 *  Gets the PivotField by the name.
	 * 
	 *  @param name the field name
	 *  @return the PivotField by the name.
	 */
	public PivotField getField(String name);

	/**
	 *  Gets the PivotField by the index.
	 * 
	 *  @param index the field index
	 *  @return the PivotField by the index.
	 */
	public PivotField getField(int index);

	/**
	 *  Gets all the PivotFields.
	 * 
	 *  @return all the PivotFields.
	 */
	public PivotField[] getFields();

	/**
	 *  Initializes the PivotDataModel. Whenever your original table model structured changed (column added, removed or
	 *  changed), you need to call this method then {@link #calculate()}. If only the data is changed, just {@link
	 *  #calculate()} is enough.
	 */
	public void initialize();

	public void updateFields();

	/**
	 *  Calculates the pivot data. Whenever you are done updating PivotField settings or the data in the original table
	 *  model is changed, you should call this method to use the new settings or new data. If you call this method after
	 *  the PivotDataModel has been set on PivotTablePane, you also need to call {@link
	 *  com.jidesoft.pivot.PivotTablePane#fieldsUpdated()} method in order for the change to take effect.
	 */
	@java.lang.SuppressWarnings("MismatchedReadAndWriteOfArray")
	public void calculate();

	/**
	 *  Checks if {@link #calculate()} is called on this pivot data model. There is no setter for this flag. Only {@link
	 *  #calculate()} will set it to true.
	 * 
	 *  @return true if {@link #calculate()} is already called.
	 */
	public boolean isCalculated();

	/**
	 *  Gets the possible values of the specified columnIndex. The values are sorted.
	 * 
	 *  @param columnIndex the column index.
	 *  @return the possible values of in the column.
	 *  @deprecated the order of the possible values returned from this method is not clear. Please use {@link
	 *  #getPossibleValues(int, java.util.Comparator)} instead. We didn't use this method anymore in our code.
	 */
	@java.lang.Deprecated
	public Object[] getPossibleValues(int columnIndex);

	/**
	 *  Gets the possible values of the specified columnIndex. The values are sorted using the comparator passed in as
	 *  parameter.
	 * 
	 *  @param columnIndex the column index
	 *  @param comparator  the comparator
	 *  @return the possible values of in the column.
	 */
	@java.lang.SuppressWarnings("RawUseOfParameterizedType")
	public Object[] getPossibleValues(int columnIndex, java.util.Comparator comparator);

	public Object[] getPossibleValues(PivotField field);

	/**
	 *  Gets the row header table model. It can be used by HeaderTable.
	 * 
	 *  @return the row header table model.
	 */
	public HeaderTableModel getRowHeaderTableModel();

	/**
	 *  Gets the corner table model. It is the table model for the top left corner but in regular PivotTablePane, this
	 *  model is not used. You need to enable it by overriding the following method in PivotTablePane and override
	 *  setupLayout to add _dataFieldsArea and _rowFieldsArea to somewhere else in order to set the _cornerTable to
	 *  UPPER_LEFT_CORNER of the JScrollPane.
	 *  <code><pre>
	 *  protected void initComponents() {
	 *      super.initComponents();
	 *     _cornerTable = new HeaderTable(this);
	 *     _cornerTable.setRowHeight(getRowHeight());
	 *     setCornerTableModel(_cornerTable, getPivotDataModel().getCornerTableModel());
	 *     _cornerTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
	 *      _cornerTable.setCellSelectionEnabled(true);
	 *      ((JideTable) _cornerTable).setColumnResizable(true);
	 *      ((JideTable) _cornerTable).setNonContiguousCellSelection(true);
	 *  }
	 *  </pre></code>
	 * 
	 *  @return the corner table model.
	 */
	public javax.swing.table.TableModel getCornerTableModel();

	/**
	 *  Gets the column header table model. It can be used by HeaderTable.
	 * 
	 *  @return the column header table model.
	 */
	public HeaderTableModel getColumnHeaderTableModel();

	/**
	 *  Gets the data table model.
	 * 
	 *  @return the data table model.
	 */
	public javax.swing.table.TableModel getDataTableModel();

	/**
	 *  Get the value to be compared.
	 * 
	 *  @param field      the data field
	 *  @param rowKeys    row value keys in the corresponding row header table
	 *  @param columnKeys column value keys in the corresponding column header table
	 *  @return the value from the cell.
	 */
	public Object getValueAt(PivotField field, Values rowKeys, Values columnKeys);

	/**
	 *  Gets the row fields. It must be called after {@link #calculate()} is called. Otherwise, it will return null.
	 * 
	 *  @return the row fields.
	 */
	public PivotField[] getRowFields();

	/**
	 *  Gets the model indices for the row fields. It must be called after {@link #calculate()} is called. Otherwise, it
	 *  will return null.
	 * 
	 *  @return the row fields.
	 */
	public int[] getRowFieldIndices();

	/**
	 *  Gets the column fields. It must be called after {@link #calculate()} is called. Otherwise, it will return null.
	 * 
	 *  @return the column fields.
	 */
	public PivotField[] getColumnFields();

	/**
	 *  Gets the model indices for the column fields. It must be called after {@link #calculate()} is called. Otherwise,
	 *  it will return null.
	 * 
	 *  @return the column fields.
	 */
	public int[] getColumnFieldIndices();

	/**
	 *  Gets the data fields. it must be called after {@link #calculate()}  or {@link #resetDataFields()} is called.
	 *  Otherwise, it will return null.
	 * 
	 *  @return the data fields.
	 */
	public PivotField[] getDataFields();

	/**
	 *  Reset data fields based on the values in every PivotField. This method should only be invoked for protection
	 *  purpose.
	 */
	public void resetDataFields();

	/**
	 *  Gets the model indices for the data fields. It must be called after {@link #calculate()} is called. Otherwise, it
	 *  will return null.
	 * 
	 *  @return the data fields.
	 */
	public int[] getDataFieldIndices();

	/**
	 *  Gets the filter fields. it must be called after {@link #calculate()} is called. Otherwise, it will return null.
	 * 
	 *  @return the filter fields.
	 */
	public PivotField[] getFilterFields();

	/**
	 *  Gets the model indices for the filter fields. It must be called after {@link #calculate()} is called. Otherwise,
	 *  it will return null.
	 * 
	 *  @return the filter fields.
	 */
	public int[] getFilterFieldIndices();

	/**
	 *  Gets the unassigned fields. it must be called after {@link #calculate()} is called. Otherwise, it will return
	 *  null.
	 * 
	 *  @return the unassigned fields.
	 */
	public PivotField[] getUnassignedFields();

	/**
	 *  Gets the model indices for the unassigned fields. It must be called after {@link #calculate()} is called.
	 *  Otherwise, it will return null.
	 * 
	 *  @return the unassigned fields.
	 */
	public int[] getUnassignedFieldIndices();

	/**
	 *  Gets the row index list at the specified row index and column index. If both indices are greater than or equal 0,
	 *  it will get the row index list as in the data table model. If the row index is less than 0 but the column index
	 *  is greater than or equal 0, it will get the row index list as in the column header table model. The negative row
	 *  index is converted to the row index relative to row header table model using - rowIndex - 1. If the column index
	 *  is less than 0 but the row index is greater than or equal 0, it will get the row index list as in the row header
	 *  table model. The negative column index is converted to the column index relative to column header table model
	 *  using - columnIndex - 1.
	 *  <p/>
	 *  Please note, the list returned from this method is used internally. If you iterate through the list and modify
	 *  the table model at the same time, the list could be changed as well. So to be safe, you should replicate the list
	 *  if you plan to modify the table model.
	 * 
	 *  @param rowIndex    the row index relative to DataTableModel.
	 *  @param columnIndex the column index relative to DataTableModel.
	 *  @return a list of integer which are row indexes to the data source that you can get using {@link
	 *  #getDataSource()}.
	 */
	public java.util.List getDataAt(int rowIndex, int columnIndex);

	/**
	 *  Gets the row index list that matches with the rowKey and columnKey.
	 * 
	 *  @param rowKey    the row compound key
	 *  @param columnKey the column compound key
	 *  @return a list of integer which are row indexes to the data source that you can get using {@link
	 *  #getDataSource()}.
	 */
	public java.util.List getDataAt(CompoundKey rowKey, CompoundKey columnKey);

	/**
	 *  Is the data field displayed on the row header area. It possibly returns true only if isColumnDataFields return
	 *  false. Otherwise data fields will display on the column header area.
	 * 
	 *  @return true or false.
	 */
	public boolean isRowDataFields();

	/**
	 *  Checks if the data field displayed on the column header area.
	 * 
	 *  @return true or false.
	 */
	public boolean isColumnDataFields();

	/**
	 *  Checks if the data fields will always displayed on the column header area. If this method returns true,
	 *  isColumnDataFields will always return true.
	 * 
	 *  @return true or false.
	 */
	public boolean isAlwaysColumnDataFields();

	/**
	 *  Sets the flag whether the data fields will always be displayed on the column header area. By default, we might
	 *  use row header area to display the data fields if there is no field in the row area.
	 * 
	 *  @param alwaysColumnDataFields true or false.
	 */
	public void setAlwaysColumnDataFields(boolean alwaysColumnDataFields);

	/**
	 *  Is the data field always displayed on the row header area. If this method returns true, isRowDataFields will
	 *  always return true as long as isColumnDataFields returns false.
	 * 
	 *  @return true or false.
	 */
	public boolean isAlwaysRowDataFields();

	/**
	 *  Sets the flag whether the data fields will always be displayed on the row header area. By default, we might use
	 *  row header area to display the data fields if there is no field in the row area.
	 * 
	 *  @param alwaysRowDataFields true or false.
	 */
	public void setAlwaysRowDataFields(boolean alwaysRowDataFields);

	public boolean isHideSingleDataField();

	public void setHideSingleDataField(boolean hideSingleDataField);

	/**
	 *  Gets the flag if empty headers are allowed.
	 * 
	 *  @return true or false.
	 */
	public boolean isAllowEmptyColumnHeader();

	/**
	 *  Sets the flag if empty column header is allowed. This is used when there are no column fields but there are data
	 *  fields. If this flag is set to false, we will display summary information on the data fields in the data table.
	 *  If set to true, the data table will be empty.
	 * 
	 *  @param allowEmptyColumnHeader true or false.
	 */
	public void setAllowEmptyColumnHeader(boolean allowEmptyColumnHeader);

	/**
	 *  Gets the flag if empty headers are allowed.
	 * 
	 *  @return true or false.
	 */
	public boolean isAllowEmptyRowHeader();

	/**
	 *  Sets the flag if empty row header is allowed. This is used when there are no row fields but there are data
	 *  fields. If this flag is set to false, we will display summary information on the data fields in the data table.
	 *  If set to true, the data table will be empty.
	 * 
	 *  @param allowEmptyRowHeader true or false.
	 */
	public void setAllowEmptyRowHeader(boolean allowEmptyRowHeader);

	/**
	 *  Checks if grand total row is visible.
	 * 
	 *  @return true if grand total row is visible. Otherwise is false.
	 */
	public boolean isShowGrandTotalForRow();

	/**
	 *  Shows grand total row. If you changed this value, you need to call {@link PivotTablePane#rowHeaderUpdated()} or
	 *  {@link PivotTablePane#bothHeadersUpdated()} if you also changed grandTotalForColumn.
	 * 
	 *  @param showGrandTotalForRow whether show grand total for row.
	 */
	public void setShowGrandTotalForRow(boolean showGrandTotalForRow);

	/**
	 *  Checks if grand total column is visible.
	 * 
	 *  @return true if grand total column is visible. Otherwise is false.
	 */
	public boolean isShowGrandTotalForColumn();

	/**
	 *  Shows grand total column. If you changed this value, you need to call {@link
	 *  PivotTablePane#columnHeaderUpdated()} or {@link PivotTablePane#bothHeadersUpdated()} if you also changed
	 *  grandTotalForRow.
	 * 
	 *  @param showGrandTotalForColumn whether show grand total for column.
	 */
	public void setShowGrandTotalForColumn(boolean showGrandTotalForColumn);

	/**
	 *  Gets the PivotCellStyleProvider.
	 * 
	 *  @return the PivotCellStyleProvider.
	 */
	public PivotCellStyleProvider getCellStyleProvider();

	/**
	 *  Sets the PivotCellStyleProvider. You can use this provider to provide CellStyle for column header table, row
	 *  header table and data table.
	 * 
	 *  @param cellStyleProvider the cell style provider.
	 */
	public void setCellStyleProvider(PivotCellStyleProvider cellStyleProvider);

	/**
	 *  Sets the PivotDataEditingProvider that is used for the row header table. You can use this provider to provide a
	 *  way to edit cell directly in pivot table. The provide will interpret the user change and modify the actual data
	 *  source.
	 * 
	 *  @param rowHeaderEditingProvider the date editing provider.
	 */
	public void setRowHeaderEditingProvider(PivotDataEditingProvider rowHeaderEditingProvider);

	/**
	 *  Gets the PivotDataEditingProvider for the row header table.
	 * 
	 *  @return the PivotDataEditingProvider.
	 */
	public PivotDataEditingProvider getRowHeaderEditingProvider();

	/**
	 *  Sets the PivotDataEditingProvider that is used for the column header table. You can use this provider to provide
	 *  a way to edit cell directly in pivot table. The provide will interpret the user change and modify the actual data
	 *  source.
	 * 
	 *  @param columnHeaderEditingProvider the date editing provider.
	 */
	public void setColumnHeaderEditingProvider(PivotDataEditingProvider columnHeaderEditingProvider);

	/**
	 *  Gets the PivotDataEditingProvider for the column header table.
	 * 
	 *  @return the PivotDataEditingProvider.
	 */
	public PivotDataEditingProvider getColumnHeaderEditingProvider();

	/**
	 *  Gets the PivotDataEditingProvider for the data table.
	 * 
	 *  @return the PivotDataEditingProvider.
	 */
	public PivotDataEditingProvider getDataEditingProvider();

	/**
	 *  Sets the PivotDataEditingProvider for the data table. You can use this provider to provide a way to edit cell
	 *  directly in pivot table. The provide will interpret the user change and modify the actual data source.
	 * 
	 *  @param dataEditingProvider the date editing provider.
	 */
	public void setDataEditingProvider(PivotDataEditingProvider dataEditingProvider);

	/**
	 *  Gets the statistics calculator used to calculate all the statistics for the pivot data model.
	 * 
	 *  @return the statistics calculator.
	 */
	public SummaryCalculator getSummaryCalculator();

	/**
	 *  Sets the statistics calculator.
	 * 
	 *  @param summaryCalculator the summary calculator.
	 */
	public void setSummaryCalculator(SummaryCalculator summaryCalculator);

	/**
	 *  Gets the summary calculator factory which will create summary calculator to calculate all the statistics for the
	 *  pivot data model.
	 * 
	 *  @return the summary calculator factory.
	 */
	public SummaryCalculatorFactory getSummaryCalculatorFactory();

	/**
	 *  Sets the statistics calculator factory. We will use this factory to create SummaryCalculator when needed.
	 * 
	 *  @param summaryCalculatorFactory the summary calculator factory.
	 */
	public void setSummaryCalculatorFactory(SummaryCalculatorFactory summaryCalculatorFactory);

	/**
	 *  Checks if the grand total is displayed as the first row.
	 * 
	 *  @return true or false.
	 */
	public boolean isDisplayGrandTotalFirstForRow();

	/**
	 *  By default, grand total is displayed at the last row in the pivot table. If you set this flag to true, it will be
	 *  displayed as the first row.
	 * 
	 *  @param displayGrandTotalFirstForRow true or false.
	 */
	public void setDisplayGrandTotalFirstForRow(boolean displayGrandTotalFirstForRow);

	/**
	 *  Checks if the grand total is displayed as the first column.
	 * 
	 *  @return true or false.
	 */
	public boolean isDisplayGrandTotalFirstForColumn();

	/**
	 *  By default, grand total is displayed at the last column in the pivot table. If you set this flag to true, it will
	 *  be displayed as the first column.
	 * 
	 *  @param displayGrandTotalFirstForColumn true or false.
	 */
	public void setDisplayGrandTotalFirstForColumn(boolean displayGrandTotalFirstForColumn);

	/**
	 *  Checks if the grand total is displayed as the first column (row).
	 * 
	 *  @return true or false.
	 *  @deprecated replaced by {@link #isDisplayGrandTotalFirstForRow()} and {@link #isDisplayGrandTotalFirstForColumn()}
	 */
	@java.lang.Deprecated
	public boolean isDisplayGrandTotalFirst();

	/**
	 *  By default, grand total is displayed at the last column (row) in the pivot table. If you set this flag to true,
	 *  it will be displayed as the first column (row).
	 * 
	 *  @param displayGrandTotalFirst true or false.
	 *  @deprecated replaced by {@link #setDisplayGrandTotalFirstForRow(boolean)} and {@link
	 *  #setDisplayGrandTotalFirstForColumn(boolean)}
	 */
	@java.lang.Deprecated
	public void setDisplayGrandTotalFirst(boolean displayGrandTotalFirst);

	/**
	 *  Checks if the pivot data will be automatically updated when the underlying table model data changes.
	 * 
	 *  @return true or false.
	 */
	public boolean isAutoUpdate();

	/**
	 *  Sets the flag if the pivot data will be automatically updated when the underlying table model data changes. By
	 *  default, it is false because in most cases people use pivot data to analyze static data. If the data changes, you
	 *  have to call {@link #calculate()} method when the underlying table model data changes. However because the data
	 *  inside pivot table is very complex, right now autoUpdate will only update correctly if the table model data
	 *  change is cellUpdated event and rowUpdated event. If you add/remove rows or columns, you still have to call
	 *  {@link #calculate()} just like before.
	 * 
	 *  @param autoUpdate true or false.
	 */
	public void setAutoUpdate(boolean autoUpdate);

	/**
	 *  Checks if the PivotDataModel is adjusting. If it is adjusting, other code should not call calculate() or update
	 *  UI automatically when a property is changed. Right now, we used in {@link com.jidesoft.pivot.PivotField#setSelectedPossibleValues(Object[])}.
	 *  By default, isAdjusting is false, so setSelectedPossibleValues will call calculate() and bothHeaderUpdated(). If
	 *  isAdjusting is true, they will be not be called.
	 * 
	 *  @return true if adjusting. Otherwise false.
	 */
	public boolean isAdjusting();

	/**
	 *  Sets the PivotDataModel to adjusting mode. If it is adjusting, PivotDataModel will not call calculate() and
	 *  update the UI when something is changed. After you set adjusting to false, PivotDataModel will then calculate().
	 *  But you will be still be responsible to call the XxxUpdated() method on PivotTablePane to update the UI.
	 * 
	 *  @param adjusting true or false.
	 */
	public void setAdjusting(boolean adjusting);

	/**
	 *  Checks if the pivot data model works in summary mode. Summary mode means instead of display the summary (such as
	 *  sum, max, min, stddev etc statistics) of the values in the data table, it displays the first value that matches
	 *  the key.
	 *  <p/>
	 *  Default is true.
	 * 
	 *  @return true or false.
	 */
	public boolean isSummaryMode();

	/**
	 *  Sets the flag for the summary mode. Summary mode means instead of display the summary (such as sum, max, min,
	 *  stddev etc statistics) of the values in the data table, it displays the first value that matches the key.
	 * 
	 *  @param summaryMode true to set to summary mode. Otherwise false.
	 *  @see #isSummaryMode()
	 */
	public void setSummaryMode(boolean summaryMode);

	/**
	 *  The cells in the data table are actually summary of multiple rows in the original table model. For example, if
	 *  the summary is SUM, we will display the sum of all the values of all the rows and display it in the cell of the
	 *  data table. However In the single value mode, whenever there are two rows that maps to the same cell, we will
	 *  delete the second row.
	 * 
	 *  @return true or false.
	 */
	public boolean isSingleValueMode();

	/**
	 *  Sets the single value mode.
	 * 
	 *  @param singleValueMode true or false.
	 */
	public void setSingleValueMode(boolean singleValueMode);

	/**
	 *  Gets the row index in the row header table which has the data from the original table model at the specified
	 *  rowIndex.
	 * 
	 *  @param rowIndex the row at the original table model.
	 *  @return the row index as in the row header table.
	 */
	public int getRowHeaderRowIndexAt(int rowIndex);

	/**
	 *  Gets the column index in the column header table which has the data from the original table model at the
	 *  specified rowIndex.
	 * 
	 *  @param rowIndex the row at the original table model.
	 *  @return the column index as in the column  header table.
	 */
	public int getColumnHeaderColumnIndexAt(int rowIndex);

	/**
	 *  Get the flag that indicating if summary values should be hidden if all its children values are hidden.
	 *  <p/>
	 *  The default value of this flag is true, which could cause incompatible behavior with release earlier than 2.8.3.
	 * 
	 *  @return true if summary values need be hidden. otherwise false.
	 */
	public boolean isHideSummaryValues();

	/**
	 *  Set the flag that indicating if summary values should be hidden if all its children values are hidden.
	 * 
	 *  @param hideSummaryValues the flag
	 *  @see #isHideSummaryValues()
	 */
	public void setHideSummaryValues(boolean hideSummaryValues);

	/**
	 *  Checks if the Values is visible.
	 * 
	 *  @param values the Values.
	 *  @return true or false True means the Values is visible.
	 */
	public boolean isValuesVisible(Values values);

	/**
	 *  Adds the values to the hidden values. For example, if you want to hide the column or row, as in PivotTableDemo,
	 *  that year is 1994, quarter is 2 and month is 6, you can call the code below to do it.
	 *  <pre><code>
	 *  addHiddenValues(new DefaultValues(new Object[]{1994, 2, 6});
	 *  </code></pre>
	 * 
	 *  @param values the Values.
	 */
	public void addHiddenValues(Values values);

	/**
	 *  Removes the values from the hidden values.
	 * 
	 *  @param values the Values.
	 */
	public void removeHiddenValues(Values values);

	/**
	 *  Clears the hidden values. In the other word, all Values will be shown on the header tables.
	 */
	public void clearHiddenValues();

	/**
	 *  Checks if there are hidden values.
	 * 
	 *  @return True if there are hidden values. Otherwise false.
	 */
	public boolean hasHiddenValues();

	/**
	 *  Gets all the hidden values.
	 * 
	 *  @return the hidden values in an array.
	 */
	public Values[] getHiddenValues();

	/**
	 *  Checks if the row is visible.
	 * 
	 *  @param values the Values
	 *  @param field  the data field
	 *  @return true or false True means the row is visible.
	 *  @since 3.5.16
	 */
	public boolean isRowVisible(Values values, PivotField field);

	/**
	 *  Adds the designated row defined by values and PivotField to the hidden values.
	 * 
	 *  @param values the Values
	 *  @param field  the data field
	 *  @since 3.5.16
	 */
	public void addHiddenRow(Values values, PivotField field);

	/**
	 *  Clears the hidden rows. In the other word, all rows will be shown on the header tables.
	 * 
	 *  @since 3.5.16
	 */
	public void clearHiddenRows();

	/**
	 *  Checks if there are hidden rows.
	 * 
	 *  @return True if there are hidden rows. Otherwise false.
	 *  @since 3.5.16
	 */
	public boolean hasHiddenRows();

	/**
	 *  Checks if the Values is visible.
	 * 
	 *  @param values the Values
	 *  @param field  the data field
	 *  @return true or false True means the Values is visible.
	 */
	public boolean isColumnVisible(Values values, PivotField field);

	/**
	 *  Adds the designated column defined by values and PivotField to the hidden values. If all data fields in one
	 *  values are hidden, addHiddenValues() will be invoked to hide the entire values.
	 *  <p/>
	 *  This method should invoke {@link #addHiddenColumn(Values, PivotField, String)} with the string as "15\t75\t75"
	 * 
	 *  @param values the Values
	 *  @param field  the data field
	 */
	public void addHiddenColumn(Values values, PivotField field);

	/**
	 *  Adds the designated column defined by values and PivotField to the hidden values. If all data fields in one
	 *  values are hidden, addHiddenValues() will be invoked to hide the entire values.
	 * 
	 *  @param values the Values
	 *  @param field  the data field
	 *  @param width  the width information before the column is hidden
	 */
	public void addHiddenColumn(Values values, PivotField field, String width);

	/**
	 *  Removes the column from the hidden columns.
	 * 
	 *  @param values the Values
	 *  @param field  the data field
	 *  @return the saved column width while hiding the column. -1 if the column width is not available.
	 */
	public String removeHiddenColumn(Values values, PivotField field);

	/**
	 *  Clears the hidden columns. In the other word, all columns will be shown on the header tables.
	 */
	public void clearHiddenColumns();

	/**
	 *  Checks if there are hidden columns.
	 * 
	 *  @return True if there are hidden columns. Otherwise false.
	 */
	public boolean hasHiddenColumns();

	/**
	 *  Get the hidden columns array.
	 *  <p/>
	 *  It will return a two-dimension array. Each row means a hidden column and has three elements. In each row, the
	 *  first element is Values and the second element is DataField while the last one is the width information before
	 *  hidden. There are three elements in the width information, which in order are minWdith, preferredWidth then
	 *  width.
	 * 
	 *  @return the hidden column array.
	 */
	public Object[][] getHiddenColumns();

	/**
	 *  Invalidate corner table model before calculate. After calculating, you should always set model for corner table
	 *  explicitly.
	 */
	public void invalidateCornerTableModel();

	/**
	 *  Invalidate data table model before calculate. After calculating, you should always set model for corner table
	 *  explicitly.
	 */
	public void invalidateDataTableModel();

	/**
	 *  Invalidate row header table model before calculate. After calculating, you should always set model for corner
	 *  table explicitly.
	 */
	public void invalidateRowHeaderTableModel();

	/**
	 *  Invalidate column header table model before calculate. After calculating, you should always set model for corner
	 *  table explicitly.
	 */
	public void invalidateColumnHeaderTableModel();

	/**
	 *  Checks the flag whether the pivot table expands to show all the values.
	 * 
	 *  @return true or false.
	 */
	public boolean isExpandByDefault();

	/**
	 *  Sets the flag whether the pivot table should expand to show all the values. By default, when calcualte() method
	 *  is called on PivotDataModel, all values are expanded so that user will see a fully expanded pivot table. You can
	 *  change it to false so that it will show only the top level summary.
	 * 
	 *  @param expandByDefault true or false.
	 */
	public void setExpandByDefault(boolean expandByDefault);

	/**
	 *  Get the flag.
	 * 
	 *  @return true if show subtotal as child. Otherwise false.
	 *  @see #setShowSubtotalAsChild(boolean)
	 */
	public boolean isShowSubtotalAsChild();

	/**
	 *  By default, the subtotal rows/columns are at the same level as the values. For example, if there is a year value
	 *  is 1994, 1994 total is at the same level as sibling. If this flag is set to true, the subtotal will become a
	 *  child of the value.
	 * 
	 *  @param showSubtotalAsChild true or false.
	 */
	public void setShowSubtotalAsChild(boolean showSubtotalAsChild);

	/**
	 *  When the subtotal is configured as automatically and there is only one item, the subtotal is redundant. By
	 *  setting this flag to false, we will hide the subtotal if there is only one item.
	 * 
	 *  @return true or false.
	 *  @since 3.3.3
	 */
	public boolean isHideRedundantSubtotal();

	/**
	 *  Sets the hideRedundantSubtotal flag.
	 * 
	 *  @param hideRedundantSubtotal the flag
	 *  @see #isHideRedundantSubtotal()
	 *  @since 3.3.3
	 */
	public void setHideRedundantSubtotal(boolean hideRedundantSubtotal);

	/**
	 *  This method is for debugging purpose, not a public API. It will print out if there is any cell span errors in the
	 *  row header and the column header table.
	 */
	@java.lang.SuppressWarnings({"UseOfSystemOutOrSystemErr", "JavaDoc"})
	public boolean verify();

	/**
	 *  Get the flag indicating that if all its row children in row header table should be kept if the summary row is
	 *  kept on data field filter.
	 *  <p/>
	 *  The default value is true which means all its row children would be kept. You could change it to false if you
	 *  want to keep all its column children of summary values in column header table.
	 *  <p/>
	 *  This flag only takes effects when at least one data field filters on summary values.
	 *  <p/>
	 *  #see {@link com.jidesoft.pivot.PivotField#isDataFieldFilterOnSummary()}
	 * 
	 *  @return true if row children of summary values are to be kept. Otherwise false.
	 */
	public boolean isKeepRowChildrenOnSummaryFilter();

	/**
	 *  Set the flag indicating that if all its row children in row header table should be kept if the summary row is
	 *  kept on data field filter.
	 *  <p/>
	 *  #see #isKeepRowChildrenOnSummaryFilter
	 * 
	 *  @param filterRowSummaryValues the flag
	 */
	public void setKeepRowChildrenOnSummaryFilter(boolean filterRowSummaryValues);

	/**
	 *  Gets the flag indicating if it's used by AggregateTable.
	 * 
	 *  @return true if it's used by AggregateTable. Otherwise false.
	 */
	public boolean isAggregateMode();

	/**
	 *  Gets the flag indicating if it's a summary layer.
	 * 
	 *  @return true if it's a summary layer. Otherwise false.
	 */
	public boolean isSummaryLayer();

	/**
	 *  Sets the flag indicating if it's a summary layer.
	 * 
	 *  @param summaryLayer the flag
	 */
	public void setSummaryLayer(boolean summaryLayer);

	/**
	 *  Gets the flag indicating if the data field should be laid out in the first level or not.
	 * 
	 *  @return true if the data field should be laid out first. Otherwise false.
	 *  @see #setLayoutDataFieldsFirst(boolean)
	 *  @since 3.4.0
	 */
	public boolean isLayoutDataFieldsFirst();

	/**
	 *  Sets the flag indicating if the data field should be laid out in the first level or not.
	 *  <p/>
	 *  By default the flag is false to keep the default behavior backward compatible.
	 * 
	 *  @param layoutDataFieldsFirst the flag
	 *  @since 3.4.0
	 */
	public void setLayoutDataFieldsFirst(boolean layoutDataFieldsFirst);

	/**
	 *  Gets the flag indicating if the IPivotDataModel should work on high performance mode.
	 * 
	 *  @return true if the IPivotDataModel should work on high performance mode. Otherwise false.
	 *  @see #setHighPerformanceMode(boolean)
	 *  @since 3.4.0
	 */
	public boolean isHighPerformanceMode();

	/**
	 *  Sets the flag indicating if the IPivotDataModel should work on high performance mode.
	 *  <p/>
	 *  By default the flag is false to keep the default behavior backward compatible.
	 *  <p/>
	 *  Setting this flag to true will improve the calculating performance. However, some features, like the expansion
	 *  state persistence, selection persistence, column width persistence, hidden value persistence, will be lost.
	 * 
	 *  @param highPerformanceMode the flag
	 *  @since 3.4.0
	 */
	public void setHighPerformanceMode(boolean highPerformanceMode);
}
