/**
 *  The package contains classes for JIDE Pivot Grid product.
 */
package com.jidesoft.pivot;


/**
 *  PivotDataEditingProvider is an interface to allow the direct modification of data in pivot table.
 *  Since there is no one to one relation between the cells in pivot table and the actual data source row.
 *  So it's up to the provider to interpret the data change and modify the data source accordingly.
 *  If in a certain case, there is no way to modify the data source, false should be returned in <code>isCellEditable</code> method.
 */
public interface PivotDataEditingProvider {

	/**
	 *  Returns true if the cell at <code>rowIndex</code> and
	 *  <code>columnIndex</code> of the DataTableModel in PivotDataModel
	 *  is editable.  Otherwise, <code>setValueAt</code> on the cell will not
	 *  change the value of that cell.
	 * 
	 *  @param dataModel   the pivot data model.
	 *  @param rowIndex    the row whose value is to be queried
	 *  @param columnIndex the column whose value is to be queried
	 *  @return true if the cell is editable
	 */
	public boolean isCellEditable(PivotDataModel dataModel, int rowIndex, int columnIndex);

	/**
	 *  This method is called when user changes the the value in the cell at <code>columnIndex</code> and
	 *  <code>rowIndex</code> to <code>aValue</code>. You should implement this method and modify the
	 *  corresponding cell or cells in data source so that the change can be persisted.
	 * 
	 *  @param dataModel   the pivot data model.
	 *  @param aValue      the new value
	 *  @param rowIndex    the row whose value is to be changed
	 *  @param columnIndex the column whose value is to be changed
	 *  @see #isCellEditable
	 */
	public void setValueAt(PivotDataModel dataModel, Object aValue, int rowIndex, int columnIndex);
}
