/**
 *  The package contains classes for JIDE Pivot Grid product.
 */
package com.jidesoft.pivot;


/**
 *  By default, PivotDataModel takes TableModel as the data source. By providing this interface, it is possible to use
 *  any compatible data as the data source for PivotDataModel. PivotDataSource also provides methods related to
 *  filtering. If you use this interface, you can totally skip the FilterableTableModel we used internally in
 *  PivotDataModel. If you can provide a more efficient filtering engine to implement this interface in the right way, it
 *  could make the pivot table faster.
 */
public interface PivotDataSource {

	/**
	 *  Gets number of fields in this data source.
	 * 
	 *  @return the field count.
	 */
	public int getFieldCount();

	/**
	 *  Gets the field name. The name should be unique in this data source.
	 * 
	 *  @param fieldIndex the field index.
	 *  @return the field name.
	 */
	public String getFieldName(int fieldIndex);

	/**
	 *  Gets the field title.
	 * 
	 *  @param fieldIndex the field index.
	 *  @return the title of the field.
	 *  @since 3.2.4
	 */
	public String getFieldTitle(int fieldIndex);

	/**
	 *  Gets the field type.
	 * 
	 *  @param fieldIndex the field index.
	 *  @return the field type.
	 */
	public Class getFieldType(int fieldIndex);

	/**
	 *  Gets the row count. If there is filtering, it should return the row count after filtering.
	 * 
	 *  @return the row count.
	 */
	public int getRowCount();

	/**
	 *  Gets the value at the specified field index and the row index.
	 * 
	 *  @param rowIndex   the row index. It is the index after filtering.
	 *  @param fieldIndex the field index.
	 *  @return the value.
	 */
	public Object getValueAt(int rowIndex, int fieldIndex);

	/**
	 *  Gets the value at the specified field index and the row index.
	 * 
	 *  @param value      the new value.
	 *  @param fieldIndex the field index.
	 *  @param rowIndex   the row index. It is the index after filtering.
	 */
	public void setValueAt(Object value, int rowIndex, int fieldIndex);

	/**
	 *  Gets the possible values for the field.
	 * 
	 *  @param fieldIndex    the field index.
	 *  @param filterField   whether the field is a filter field. If true, possible values will all values as in the
	 *                       original data source. If false, the rows will be filtered first based on the {@link
	 *                       #setFilter(Object[],int,boolean)} and then collect all the possible values.
	 *  @param isNullAllowed whether null value is allowed.
	 *  @return the possible values.
	 */
	public java.util.Set getPossibleValues(int fieldIndex, boolean filterField, boolean isNullAllowed);

	/**
	 *  Tells the data source to clear all existing filters so that we can start to use {@link
	 *  #setFilter(Object[],int,boolean)} method to set the filters.
	 */
	public void clearFilters();

	/**
	 *  Get if the data source has filter applying on it.
	 * 
	 *  @return true if there are filters. Otherwise false.
	 */
	public boolean hasFilter();

	/**
	 *  Sets the possible values for the field.
	 * 
	 *  @param values      the possible values.
	 *  @param filterField whether the field is a filter field.
	 *  @param fieldIndex  the field index.
	 */
	public void setFilter(Object[] values, int fieldIndex, boolean filterField);

	/**
	 *  Sets the deselected possible values for the field.
	 * 
	 *  @param values      the deselected possible values.
	 *  @param filterField whether the field is a filter field.
	 *  @param fieldIndex  the field index.
	 */
	public void setExcludeFilter(Object[] values, int fieldIndex, boolean filterField);

	/**
	 *  Sets the Filter for the field.
	 * 
	 *  @param filter      the Filter.
	 *  @param filterField whether the field is a filter field.
	 *  @param fieldIndex  the field index.
	 */
	public void setFilter(Filter filter, int fieldIndex, boolean filterField);

	/**
	 *  Tells the data source to apply the filters after all filters have been set using {@link
	 *  #setFilter(Object[],int,boolean)} method. This method tells you the field indices of each area. In
	 *  TableModelPivotDataSource implementation, we simply call {@link #applyFilters()} as that implementation doesn't
	 *  need to know the field indices.
	 * 
	 *  @param rowFieldIndices    the field indices for row fields.
	 *  @param columnFieldIndices the field indices for column fields.
	 *  @param dataFieldIndices   the field indices for data fields.
	 *  @param filterFieldIndices the field indices for filter fields.
	 */
	public void applyFilters(int[] rowFieldIndices, int[] columnFieldIndices, int[] dataFieldIndices, int[] filterFieldIndices);

	/**
	 *  Tells the data source to apply the filters after all filters have been set using {@link
	 *  #setFilter(Object[],int,boolean)} method. If you need to know the field indices of each area, you can override
	 *  other applyFilters method with four int[] parameters.
	 */
	public void applyFilters();

	/**
	 *  Gets the row index after filtering so that we can use {@link #getValueAt(int,int)} or {@link
	 *  #setValueAt(Object,int,int)} methods.
	 * 
	 *  @param actualRowIndex the actual row index.
	 *  @return the filtered row index. -1 of the row is filtered.
	 */
	public int getFilteredRowIndex(int actualRowIndex);

	/**
	 *  Gets the actual row index in the original table model. You can use this to get the actual index based on the
	 *  visual row index you got in {@link PivotDataModel#getDataAt(int, int)} method.
	 * 
	 *  @param visualRowIndex the visual row index.
	 *  @return the actual row index. -1 of the row is filtered.
	 */
	public int getActualRowIndex(int visualRowIndex);

	/**
	 *  Adds the table model listener. Although there is no table model concept in the data source, TableModelEvent can
	 *  still be used to indicate data change in the data source. Please note, the row index is the index as in the
	 *  original data without filtering. When you fire the table model event, you can leave the source of it to null.
	 * 
	 *  @param listener the table model listener to be added.
	 */
	public void addTableModelListener(javax.swing.event.TableModelListener listener);

	/**
	 *  Removes the table model listener that was added earlier.
	 * 
	 *  @param listener the table model listener to be removed.
	 */
	public void removeTableModelListener(javax.swing.event.TableModelListener listener);
}
