/**
 *  The package contains classes for JIDE Pivot Grid product.
 */
package com.jidesoft.pivot;


public interface PivotValueProvider {

	/**
	 *  Retrieve the value for the data field with the designated row values and column values.
	 * 
	 *  @param dataField    the data field
	 *  @param rowValues    row values in the corresponding row header table
	 *  @param columnValues column values in the corresponding column header table
	 *  @return the value from the cell.
	 */
	public Object getValueAt(PivotField dataField, Values rowValues, Values columnValues);

	/**
	 *  Retrieve the possible values for the designated field.
	 *  <p/>
	 *  In general, you need consider the current filtering to return possible values appropriately.
	 * 
	 *  @param field the pivot field
	 *  @return possible values array.
	 */
	public Object[] getPossibleValues(PivotField field);

	/**
	 *  Get row key array.
	 * 
	 *  @return the row key array.
	 */
	public CompoundKey[] getRowKeys();

	/**
	 *  Get column key array.
	 * 
	 *  @return the column key array.
	 */
	public CompoundKey[] getColumnKeys();

	/**
	 *  Get the statistics layer. By default, it should return {@link com.jidesoft.pivot.PivotConstants#STATISTICS_CALCULATION_DEFAULT}.
	 *  So that the PivotValueProvider itself will conduct the statistics. However, for running percentage or daily average,
	 *  you probably need return other values so that {@link com.jidesoft.pivot.CalculatedPivotDataModel} would calculate
	 *  the summary itself.
	 * 
	 *  @param rowValues    the row values
	 *  @param columnValues the column values
	 *  @param dataField     the data field
	 *  @return the statistics layer.
	 */
	public int getStatisticsType(Values rowValues, Values columnValues, PivotField dataField);

	/**
	 *  Pass fields information to PivotValueProvider for calculation.
	 *  @param rowFields    the row fields array
	 *  @param columnFields the column fields array
	 *  @param dataFields   the data fields array
	 *  @param filterFields the filter fields array
	 */
	public void setPivotFields(PivotField[] rowFields, PivotField[] columnFields, PivotField[] dataFields, PivotField[] filterFields);

	/**
	 *  Ask PivotValueProvider to start calculate. The calculation could be done in a separate thread so it does not mean that
	 *  the calculation has completed after invoking this method. The pivot value user will wait for the PivotValueProviderEvent
	 *  to get the information that the calculation has completed.
	 *  <p/>
	 *  The upper level could call this method in any time. You need consider the performance on this scenario.
	 */
	public void forceCalculate();

	/**
	 *  Returns if PivotValueProvider is calculating.
	 *  @return true if it is calculating. Otherwise false.
	 */
	public boolean isCalculating();

	/**
	 *  Add PivotValueProviderListener so that the listener could know when calculation starts or ends.
	 *  @param l the listener
	 */
	public void addPivotValueProviderListener(PivotValueProviderListener l);

	/**
	 *  Remove registered PivotValueProviderListener.
	 *  @param l the listener
	 */
	public void removePivotValueProviderListener(PivotValueProviderListener l);

	/**
	 *  Checks if the pivot data model works in summary mode. Summary mode means instead of display the summary (such as
	 *  sum, max, min, stddev etc statistics) of the values in the data table, it displays the first value that matches
	 *  the key.
	 *  <p/>
	 *  Default is true.
	 * 
	 *  @return true or false.
	 */
	public boolean isSummaryMode();

	/**
	 *  Sets the flag for the summary mode. Summary mode means instead of display the summary (such as sum, max, min,
	 *  stddev etc statistics) of the values in the data table, it displays the first value that matches the key.
	 * 
	 *  @param summaryMode true to set to summary mode. Otherwise false.
	 *  @see #isSummaryMode()
	 */
	public void setSummaryMode(boolean summaryMode);

	/**
	 *  The cells in the data table are actually summary of multiple rows in the original table model. For example, if
	 *  the summary is SUM, we will display the sum of all the values of all the rows and display it in the cell of the
	 *  data table. However In the single value mode, whenever there are two rows that maps to the same cell, we will
	 *  delete the second row.
	 * 
	 *  @return true or false.
	 */
	public boolean isSingleValueMode();

	/**
	 *  Sets the single value mode.
	 * 
	 *  @param singleValueMode true or false.
	 */
	public void setSingleValueMode(boolean singleValueMode);
}
